import java.util.concurrent.atomic.AtomicInteger;
import java.sql.Timestamp;
import java.util.Properties;
import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;

public class MarketProducer {

    public static String[] QUOTE_COLS = {"time", "ticker", "bid", "bid_size", "ask", "ask_size" };
    public static String[] TRADE_COLS = {"time", "ticker", "price", "size" };
    public static String KAFKA_SER = "org.apache.kafka.common.serialization";

    public static String asJSON (Object[] data, String[] columnNames) {
        StringBuilder sb = new StringBuilder();
        Boolean hasColumns = columnNames.length > 0;
        sb.append("[");
        if (hasColumns) {
            sb.append("{");
        }

        for (int i = 0; i < data.length; i++) {
            if (hasColumns) {
                sb.append("\"" + columnNames[i] + "\":");
            }
            if (data[i] instanceof String || data[i] instanceof Timestamp) {
                sb.append("\"" + data[i] + "\"");
            } else {
                sb.append("" + data[i]);
            }
            sb.append(",");
        }
        sb.setLength(sb.length() - 1);
        if (hasColumns) {
            sb.append("}");
        }
        sb.append("]");
        return sb.toString();
    }

    public static String getenv (String name, String fallback) {
        String value = System.getenv(name);
        return value == null ? fallback : value;
    }

    public static void setProp (Properties props, String key, Object value) {
        System.out.println("Setting '" + key + "' = '" + value + "'");
        props.put(key, value);
    }

    public static void produceJsonMessages(Properties props, MarketGenerator generator, int waitTime, int sampleIters, Boolean prefixColumns) throws Exception {
        System.out.println("Initializing Kafka Producer");
        KafkaProducer<String, String> producer = new KafkaProducer<String, String>(props);

        Object[][][] history;
        Object[][]   quotes, trades;

        String payload;
        int quoteBytes = 0, tradeBytes = 0, loop = 0;
        long start = 0, end = 0;
        double period = 0;
        String[] quote_cols = (prefixColumns) ? QUOTE_COLS : new String[0];
        String[] trade_cols = (prefixColumns) ? TRADE_COLS : new String[0];

        while (true) {
            history    = generator.simulate(1, 10);
            quotes     = history[0];
            trades     = history[1];

            for (int i = 0; i < quotes.length; i++) {
                payload = asJSON(quotes[i], quote_cols);
                quoteBytes += payload.length();
                producer.send(new ProducerRecord<String, String>("quotes", ""+quotes[i][1], payload));
            }

            for (int i = 0; i < trades.length; i++) {
                payload = asJSON(trades[i], trade_cols);
                tradeBytes += payload.length();
                producer.send(new ProducerRecord<String, String>("trades", ""+trades[i][1], payload));
            }

            if (waitTime != 0) { Thread.sleep(waitTime); }

            if (loop == 0) {
                end = System.currentTimeMillis();
                period = (1024*1024) * (end - start) / 1000.0;
                System.out.printf("Speed quotes=%.3fMb/s trades=%.3fMb/s%n", quoteBytes / period, tradeBytes / period);
                quoteBytes = 0;
                tradeBytes = 0;
                start = System.currentTimeMillis();
            }

            // only print every few iterations
            loop = (loop + 1) % sampleIters;
        }
    }

    public static void main(String[] args) throws Exception {
        int waitTime           = Integer.valueOf(getenv("KX_KAFKA_WAIT_TIME", "1000")),
            retries            = Integer.valueOf(getenv("KX_KAFKA_RETRIES", "0")),
            batchSize          = Integer.valueOf(getenv("KX_KAFKA_BATCH_SIZE", "16384")),
            lingerMS           = Integer.valueOf(getenv("KX_KAFKA_LINGER_MS", "1")),
            bufferMem          = Integer.valueOf(getenv("KX_KAFKA_BUFFER_MEMORY", "33554432")),
            sampleIters        = Integer.valueOf(getenv("KX_KAFKA_SAMPLE_ITERATIONS", "10"));
        boolean useSSL         = Boolean.valueOf(getenv("KX_KAFKA_USE_SSL", "false")),
                includeColumns = Boolean.valueOf(getenv("KX_INCLUDE_JSON_COLUMNS", "false"));
        String brokers         = getenv("KX_KAFKA_BROKERS", "localhost:9092"),
               acks            = getenv("KX_KAFKA_ACKS", "0"),
               truststore      = getenv("KX_KAFKA_TRUSTSTORE_LOCATION", "/opt/kx/certs/kafka.truststore.jks"),
               keystore        = getenv("KX_KAFKA_KEYSTORE_LOCATION", "/opt/kx/certs/kafka.keystore.jks"),
               certpass        = getenv("KX_KAFKA_CERTIFICATE_PASSWORD", ""),
               endpointID      = getenv("KX_KAFKA_ENDPOINT_IDENTIFICATION_ALGORITHM", ""),
               sslVerify       = getenv("KX_KAFKA_SSL_CERTIFICATE_VERIFICATION", "true"),
               registryUrl     = getenv("KX_KAFKA_SCHEMA_REGISTRY_URL", "http://schema-registry:8081");

        Properties props = new Properties();
        setProp(props, "bootstrap.servers", brokers);
        setProp(props, "acks", acks);
        setProp(props, "retries", retries);
        setProp(props, "batch.size", batchSize);
        setProp(props, "linger.ms", lingerMS);
        setProp(props, "buffer.memory", bufferMem);
        setProp(props, "key.serializer", KAFKA_SER + ".StringSerializer");
        setProp(props, "value.serializer", KAFKA_SER + ".StringSerializer");

        // SSL Configuration
        if (useSSL) {
            setProp(props, "security.protocol", "SSL");
            setProp(props, "ssl.truststore.location", truststore);
            setProp(props, "ssl.truststore.password", certpass);
            setProp(props, "ssl.keystore.location", keystore);
            setProp(props, "ssl.keystore.password", certpass);
            setProp(props, "ssl.key.password", certpass);
            setProp(props, "ssl.endpoint.identification.algorithm", endpointID);
            setProp(props, "enable.ssl.certificate.verification", sslVerify);
        }

        System.out.println("Setting 'interval' = '" + waitTime +  "'");

        MarketGenerator generator = new MarketGenerator(10, 10000);

        System.out.println("Starting generator...");
        // Warning: infinite loop!
        produceJsonMessages(props, generator, waitTime, sampleIters, includeColumns);
    }
}
